////////////////////////////////////////////////////////////////////////////////
//
// Copyright (c) since 1999. VASCO DATA SECURITY
// All rights reserved. http://www.vasco.com
//
////////////////////////////////////////////////////////////////////////////////

#import <Foundation/Foundation.h>

NS_ASSUME_NONNULL_BEGIN

@class CDDCParams;
@class ActivationParams;
@class NotificationRegistrationParams;
@class ChangePasswordParams;
@class LocalAuthenticationParams;
@class LocalTransactionParams;
@class CDDCMessageParams;
@protocol CDDCDataFeederProtocol;
@protocol UserManagerProtocol;
@protocol OrchestrationErrorDelegate;
@protocol OrchestrationWarningDelegate;
@protocol RemoteAuthenticationDelegate;
@protocol RemoteTransactionDelegate;
@protocol UserAuthenticationDelegate;

/**
 * <p>The {@link OrchestrationProtocol} protocol is the entry point of the Orchestration SDK.
 * It allows the execution of orchestration commands providing seamless two-factor authentication.</p>
 * <h3>Creation</h3>
 * <p>You can create an orchestrator using the internal class {@link OrchestratorBuilder}.
 * This builder will provide you with the minimal required configuration for the orchestrator.</p>
 * <p>Subsequent configurations can be done during the lifecycle of the {@link OrchestrationProtocol} object
 * by using the related setters.</p>
 * <h3>Provisioning</h3>
 * <p>In order to provision the SDK with a DIGIPASS instance, a call to {@link #startActivation:(ActivationParams*)} must be performed.<br/>
 * <h3>Notification registration</h3>
 * <p>In order to provision the SDK with a DIGIPASS instance, a call to {@link #startNotificationRegistration:(NotificationRegistrationParams*)} must be performed.<br/>
 * <h3>Commands</h3>
 * All subsequent orchestration commands transmission shall be done through the {@link #execute:(NSString *)} method.</p>
 */

NS_REFINED_FOR_SWIFT
@protocol OrchestrationProtocol <NSObject>

/**
 * Executes an Orchestration Command provided by the server.
 *
 * @param command the Orchestration Command provided by the server, as an hexadecimal string.
 */
- (void)execute:(NSString *)command;

/**
 * Starts the activation process.
 */
- (void)startActivation:(ActivationParams *)activationParams;

/**
 * Starts registering for push notifications.
 */
- (void)startNotificationRegistration:(NotificationRegistrationParams *)notificationRegistrationParams;

/**
 * Starts change password flow.
 */
- (void)startChangePassword:(ChangePasswordParams *)changePasswordParams;

/**
 * Starts a local authentication process.
 * This method is used to generate a one-time password using a specified protection type and an optional challenge.
 */
- (void)startLocalAuthentication:(LocalAuthenticationParams *)localAuthenticationParams;

/**
 * Starts a local transaction process.
 * This method is used to generate a signature from specified data fields and using a specified protection type.
 * @param localTransactionParams a {@link LocalTransactionParams} object
 */
- (void)startLocalTransaction:(LocalTransactionParams *)localTransactionParams;

/**
 * Provides a {@link UserManagerProtocol} object that allows manipulation of users and DIGIPASSes.
 *
 * @return a {@link UserManagerProtocol} object
 */
- (id<UserManagerProtocol>)userManager;

/**
 * Provides a {@link CDDCDataFeederProtocol} object that allows integrator to set CDDC data values .
 * @return a {@link CDDCDataFeederProtocol} object
 */
- (id <CDDCDataFeederProtocol>)cddcDataFeeder;

/**
 * Generate a CDDC Message object
 * @param cddcMessageParams {@link CDDCMessageParams} object
 * @param error a NSError raised if there is an error during the process.
 * @return nil if an error occured, or a string with the CDDC data in clear text or encrypted.
*/
- (NSString *_Nullable)cddcMessage:(CDDCMessageParams *)cddcMessageParams error:(NSError **)error;

/**
 * Sets the delegate to be notified about remote authentication events.
 */
- (void)setRemoteAuthenticationDelegate:(id<RemoteAuthenticationDelegate>)remoteAuthenticationDelegate;

/**
 * Sets the delegate to be notified about remote transaction validation events.
 */
- (void)setRemoteTransactionDelegate:(id<RemoteTransactionDelegate>)remoteTransactionDelegate;

/**
 * Sets the {@link UserAuthenticationDelegate} to override the way a user authentication will be displayed.
 *
 * @param userAuthenticationDelegate a {@link UserAuthenticationDelegate} object
 * @param types a list of {@link UserAuthenticationType}, the types of user authentication that will be overridden
 *
 */
- (void)setUserAuthenticationDelegate:(id<UserAuthenticationDelegate>)userAuthenticationDelegate forUserAuthenticationTypes:(NSArray<NSNumber *> *)types;

/**
 * Sets the delegate called when errors happen.
 */
- (void)setErrorDelegate:(id<OrchestrationErrorDelegate>) orchestrationErrorDelegate;

/**
 * Sets the delegate called when warnings happen.
 */
- (void)setWarningDelegate:(id <OrchestrationWarningDelegate>)orchestrationWarningDelegate;

@end

NS_ASSUME_NONNULL_END
